#ifndef _DX_VOS_STDIO_H
#define _DX_VOS_STDIO_H

/*! \file DX_VOS_stdio.h
    \brief This module enables standard IO operations 
*/

#include "DX_VOS_File.h"

#ifdef __cplusplus
extern "C"
{
#endif

/* Prints a string to the standard output. */
IMPORT_C void  DX_VOS_PrintString(const DxChar *aStr);

/*! Print formatted output to the standard output stream. */
IMPORT_C void  DX_VOS_Printf(const DxChar *format,...);

/*! Print formatted output to the standard output stream. */
IMPORT_C void  DX_VOS_VPrintf (const DxChar *format, DX_VA_LIST ap);

/*! Writes formatted data to a string.
	\return 
	- DX_SUCCESS - on success
	- DX_BUFFER_IS_NOT_BIG_ENOUGH if not all the output could be written to the storage location.
	  in this case only the data that can be fit in will be written 
*/
IMPORT_C DxStatus DX_VOS_SPrintf(
	DxChar *aBuf,			/*!< [out] Pointer to storage location for output */
	DxUint aBuffSize,		/*!< [in] size in bytes of storage location */
	const DxChar *format,	/*!< [in] format string. Supports: %d, %u, %l, %c, %ul, %x, %X, %s */
	...);

/*! Writes formatted data to a string. See DX_VOS_SPrintf() for details. */
IMPORT_C DxStatus DX_VOS_VSPrintf(
	 DxChar *aBuf,		
	 DxUint aBuffSize,	
	 const DxChar *format,	
	 DX_VA_LIST arg_list
	 );

#ifndef DX_NO_FILE_UTILS
/*! Writes formatted data to the file. 
\return 
- DX_SUCCESS - on success
- DX_VOS_FILE_ERROR - if there were some problems writing the data to the disk.
*/
IMPORT_C DxStatus DX_VOS_FPrintf(
	DxVosFile aFile,  /*!< [in] Pointer to output file. */
	const DxChar *format,	/*!< [in] format string. */
	...);

/*! Writes formatted data to a file. See DX_VOS_FPrintf() for details. */
IMPORT_C DxStatus DX_VOS_VFPrintf(
	DxVosFile aFile,  /*!< [in] Pointer to output file. */
	const DxChar *format,	/*!< [in] format string. */
	DX_VA_LIST arg_list);

#endif

/*! Reads a "\r" or "\n" terminated string from the standard input */
IMPORT_C DxStatus  DX_VOS_GetString(
	DxChar *aBuf,		/*!< [out] Pointer to buffer that will hold the string that is read */
	DxUint aBuffSize  /*!< [in]  Size in bytes of output buffer */ 
	);

/*! Retrieves a character from the console without echo. 
 \return the character that was read. */
IMPORT_C DxInt DX_VOS_GetCh(void);

/*! Redirects the standard output (stdout) to the specified file. */
IMPORT_C DxStatus DX_VOS_RedirectStdoutToFile(const DxChar* fileName);

/*! Cancels a redirection previously done by DX_VOS_RedirectStdoutToFile() */
IMPORT_C void DX_VOS_CancelStdOutRedirection(void);

/*! Flushes output to stdout */
IMPORT_C DxStatus DX_VOS_UpdateStdout(void);


#ifdef DX_USE_LEGACY_VOS
#define DX_VOS_Fprintf	DX_VOS_FPrintf
#define DX_VOS_Getch DX_VOS_GetCh
IMPORT_C void DX_VOS_ScanfInt(int* aInt_ptr); 
#endif
#ifdef DX_USE_INSECURE_LEGACY_VOS
IMPORT_C DxUint32  DX_VOS_Sprintf(DxInt8 *aBuf_ptr,const DxInt8 *aStr_ptr,...);
IMPORT_C DxUint32  DX_VOS_FScanfString(DxVosFileHandle_ptr aFile_ptr,DxInt8 *aStr_ptr);
#define DX_VOS_ScanfString(aString_ptr) DX_VOS_GetString(aString_ptr, DX_MAX_UINT) 

#endif

#ifdef  __cplusplus
}
#endif


#endif /* ifndef _DX_VOS_STDIO_H */









